#!/bin/bash

# make sure that phonopy is in your python path or activate your phonopy conda
# install phonopy: https://atztogo.github.io/phonopy/install.html
conda activate phonopy

# these are additional scripts to directly plot the IR spectrum
# https://github.com/JMSkelton/Phonopy-Spectroscopy
export PYTHONPATH=${PYTHONPATH}:/PATH-TO/Phonopy-Spectroscopy
export PATH=${PATH}:/PATH-TO/Phonopy-Spectroscopy/Scripts


PSEUDO='./ONCV-abinit'
ecutwfc=80
$NATOM=14

for inputDFT in 'PBE' 'HSE' 'pbesol' 'pbe0' 'b3lyp'; do


mkdir $inputDFT
cd $inputDFT

cat >relax.in <<EOF
&control
 calculation= 'relax',
 prefix     = 'benzene',
 outdir     = './tmp',
 pseudo_dir = '$PSEUDO',
 tprnfor    = .true. 
 forc_conv_thr = 1e-3
/
&system
 ibrav       = 0,
 nat         = 12,
 ntyp        = 2,
 ecutwfc     = $ecutwfc 
 occupations = 'fixed'
 !vdw_corr    = 'd3'
 nosym = .true.
 input_dft= '$inputDFT'
/
&electrons
 conv_thr    = 1.d-9,
 mixing_beta = 0.3
 electron_maxstep=5000
/
&IONS
/

CELL_PARAMETERS {bohr}
    36.5105671080000000    0.0000000000000000    0.0000000000000000
     0.0000000000000000   37.7372589790000000    0.0000000000000000
     0.0000000000000000    0.0000000000000000   28.3553875240000000

ATOMIC_SPECIES
C    15.999   C_ONCV_PBE_sr.upf
H     1.008   H_ONCV_PBE_sr.upf

K_POINTS gamma



ATOMIC_POSITIONS crystal
C  0.4375674584519648  0.5353955196219601  0.5000000000000000 0 0 0
C  0.4371004407158547  0.4653870254803805  0.5000000000000000
C  0.5004666800110046  0.5700082491454838  0.5000000000000000
C  0.5628995592841450  0.5346129745196195  0.5000000000000000
C  0.5624325415480353  0.4646049813045011  0.5000000000000000
C  0.4995333199889953  0.4299917508545162  0.5000000000000000
H  0.3888373736372454  0.5630219938805826  0.5000000000000000
H  0.5008312706391596  0.6246512026493289  0.5000000000000000
H  0.3880063187498117  0.4383710164372399  0.5000000000000000
H  0.4991687293608402  0.3753487973506711  0.5000000000000000
H  0.6119936812501884  0.5616294844892213  0.5000000000000000
H  0.6111626263627548  0.4369780061194173  0.5000000000000000

EOF

mpirun -np 6 pw.x  < relax.in > relax.out

awk '/Begin/{flag=1;next}/End/{flag=0}flag'  relax.out  >>coord.dat

echo "creating the displacement files"

cat >scf.in <<EOF
&control
 calculation= 'scf',
 prefix     = 'benzene',
 outdir     = './tmp',
 pseudo_dir = '$PSEUDO',
 tprnfor    = .true.
/
&system
 ibrav       = 0,
 nat         = 12,
 ntyp        = 2,
 ecutwfc     = 80 
 occupations = 'fixed'
 input_dft= '$inputDFT'
/
&electrons
 conv_thr    = 1.d-9,
 mixing_beta = 0.3
 electron_maxstep=5000
/
&IONS
/


CELL_PARAMETERS {bohr}
    36.5105671080000000    0.0000000000000000    0.0000000000000000
     0.0000000000000000   37.7372589790000000    0.0000000000000000
     0.0000000000000000    0.0000000000000000   28.3553875240000000


ATOMIC_SPECIES
C    15.999   C_ONCV_PBE_sr.upf
H     1.008   H_ONCV_PBE_sr.upf

K_POINTS automatic
 1 1 1 0 0 0

ATOMIC_POSITIONS crystal
$(awk '{printf("%s %4.8f %4.8f %4.8f\n", $1, $2, $3, $4) }' coord.dat |tail -12)
EOF


phonopy --qe -d --dim="1 1 1" -c scf.in -v


mpirun -np 6 pw.x   < scf.in > scf.out



rm -rf tmp


for i in $(seq -w 001 048); do

echo "now running $i out of 84"

cat >scf.in <<EOF
&control
 calculation= 'scf',
 prefix     = 'benzene',
 pseudo_dir = '$PSEUDO',
 outdir     = './tmp',
 tprnfor    = .true.
/
&system
 ibrav       = 0,
 nat         = 12,
 ntyp        = 2,
 ecutwfc     = $ecutwfc 
 occupations = 'fixed'
 !vdw_corr    = 'd3'
 input_dft= '$inputDFT'
/
&electrons
 conv_thr    = 1.d-10,
 mixing_beta = 0.3
 electron_maxstep=5000
/
&IONS
/


CELL_PARAMETERS {bohr}
    36.5105671080000000    0.0000000000000000    0.0000000000000000
     0.0000000000000000   37.7372589790000000    0.0000000000000000
     0.0000000000000000    0.0000000000000000   28.3553875240000000

ATOMIC_SPECIES
C    15.999   C_ONCV_PBE_sr.upf
H     1.008   H_ONCV_PBE_sr.upf

K_POINTS gamma


ATOMIC_POSITIONS crystal
$(tail -12  supercell-$i.in)

EOF

mpirun -np 6 pw.x  <scf.in >supercell-$i.out


done

echo "PWSCF all done"


#write BORN file (calculated on GGA level)

cat >BORN <<EOF
default value
1.027008 0.000000 0.000000 0.000000 1.026976 0.000000 0.000000 0.000000 1.014631   
0.064060 -0.001060 0.000000 -0.001070 -0.100510 0.000000 0.000000 0.000000 0.130680   
0.053360 0.057850 0.000000 0.057320 -0.015770 0.000000 0.000000 0.000000 -0.130190   
-0.048040 0.000600 0.000000 0.000880 0.086200 0.000000 0.000000 0.000000 -0.130200   
0.052840 0.057890 0.000000 0.057490 -0.016170 0.000000 0.000000 0.000000 -0.130210   
0.051410 -0.058470 0.000000 -0.058230 -0.014300 0.000000 0.000000 0.000000 -0.130090   
-0.048540 0.000760 0.000000 0.000920 0.085820 0.000000 0.000000 0.000000 -0.130200   
-0.057880 0.071700 0.000000 0.071490 0.022220 0.000000 0.000000 0.000000 0.130840   
0.063970 -0.001060 0.000000 -0.001070 -0.100840 0.000000 0.000000 0.000000 0.130640   
-0.060630 -0.071110 0.000000 -0.070890 0.023680 0.000000 0.000000 0.000000 0.130640   
0.064060 -0.001060 0.000000 -0.001070 -0.100510 0.000000 0.000000 0.000000 0.130680   
-0.060370 -0.071010 0.000000 -0.070790 0.023820 0.000000 0.000000 0.000000 0.130670   
-0.058600 0.072090 0.000000 0.071870 0.021940 0.000000 0.000000 0.000000 0.130650   
EOF


# collect the force constants

phonopy --qe -f  supercell-{001..048}.out --nac

cat >dos.in <<EOF
DIM = 1 1 1
MP =  2 2 2
GAMMA_Center = .TRUE.
SIGMA = 0.1
EOF

phonopy --dim="1 1 1" --fc-symmetry --mesh="1 1 1" --eigenvectors --qe -c scf.in --nac

phonopy --qe -c scf.in -p -s dos.in

phonopy --dim="1 1 1" --fc-symmetry --mesh="1 1 1" --eigenvectors --qe -c scf.in

phonopy-ir --linewidth=16.5 --spectrum_range="0.0 5000"

rm -rf tmp
rm -rf supercell-*.in

cd ..

done


